/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "udmsearch.h"
#include "udm_xmalloc.h"

#if (WIN32|WINNT)
#include <io.h>
#include <process.h>

#define udm_mutex_t		CRITICAL_SECTION
#define InitMutex(x)		InitializeCriticalSection(x)
#define DestroyMutex(x)		DeleteCriticalSection(x)
#define UDM_MUTEX_LOCK(x)	EnterCriticalSection(x)
#define UDM_MUTEX_UNLOCK(x)	LeaveCriticalSection(x)
#else
#include <unistd.h>
#ifdef HAVE_PTHREAD
#include <pthread.h>
#define udm_mutex_t		pthread_mutex_t
#define InitMutex(x)		pthread_mutex_init(x,NULL)
#define DestroyMutex(x)		pthread_mutex_destroy(x)
#define UDM_MUTEX_LOCK(x)	pthread_mutex_lock(x)
#define UDM_MUTEX_UNLOCK(x)	pthread_mutex_unlock(x)
#else
#define udm_mutex_t		int
#define InitMutex(x)		*(x)=0
#define DestroyMutex(x)
#define UDM_MUTEX_LOCK(x)
#define UDM_MUTEX_UNLOCK(x)
#endif
#endif

static unsigned int seconds =0; /* To sleep between documents	*/
static int flags	=0; /* For indexer			*/
static int total_threads=0; /* Total threads number		*/
static int sleep_threads=0; /* Number of sleepping threads      */
static int next_thread  =1; /* Handle number for indexer	*/
static int max_index_time=-1;
static int cur_url_number=0;
static int log2stderr=1;
static const char *config_name= UDM_CONF_DIR UDMSLASHSTR "indexer.conf";
static int add_servers=UDM_FLAG_ADD_SERV;
static int load_langmaps=UDM_FLAG_LOAD_LANGMAP;

static UDM_ENV Conf;

static udm_mutex_t mutex[UDM_LOCK_MAX];

static void InitMutexes(void){
	int i;
	for(i=0;i<UDM_LOCK_MAX;i++){
		InitMutex(&mutex[i]);
	}
}
static void DestroyMutexes(void){
	int i;
	for(i=0;i<UDM_LOCK_MAX;i++){
		DestroyMutex(&mutex[i]);
	}
}

/* CALL-BACK Locking function */
static void UdmLockProc(UDM_AGENT *A,int command,int type,const char *fn,int ln){
#ifdef DEBUG_LOCK
	fprintf(stderr,"[%d] Try %s\t%d\t%s\t%d\n",
		A?A->handle:-1,(command==UDM_LOCK)?"lock":"unlock",
		type,fn,ln);
#endif
	switch(command){
		case UDM_LOCK:
			UDM_MUTEX_LOCK(&mutex[type]);
			break;
		case UDM_UNLOCK:
			UDM_MUTEX_UNLOCK(&mutex[type]);
			break;
	}
#ifdef DEBUG_LOCK
	fprintf(stderr,"[%d] %s\t%d\t%s\t%d\n",
		A?A->handle:-1,(command==UDM_LOCK)?"locked":"unlocked",
		type,fn,ln);
#endif
}

static int ShowStatistics(UDM_AGENT *Indexer){
	int		res;
	UDM_STATLIST	Stats;
	size_t		snum;
	UDM_STAT	Total;
	
	bzero(&Total,sizeof(Total));
	res=UdmStatAction(Indexer,&Stats);
	printf("\n          Database statistics\n\n");
	printf("%10s %10s %10s\n","Status","Expired","Total");
	printf("   -----------------------------\n");
	for(snum=0;snum<Stats.nstats;snum++){
		UDM_STAT	*S=&Stats.Stat[snum];
		printf("%10d %10d %10d %s\n",S->status,S->expired,S->total,UdmHTTPErrMsg(S->status));
		Total.expired+=S->expired;
		Total.total+=S->total;
	}
	printf("   -----------------------------\n");
	printf("%10s %10d %10d\n","Total",Total.expired,Total.total);
	printf("\n");
	UDM_FREE(Stats.Stat);
	return(res);
}

/* CallBack Func for Referers*/
static void UdmRefProc(int code, const char *url, const char * ref){
	printf("%d %s %s\n",code,url,ref);
}

static int ShowReferers(UDM_AGENT * Indexer){
int res;
	printf("\n          URLs and referers \n\n");
	res=UdmURLAction(Indexer,NULL,UDM_URL_ACTION_REFERERS,Indexer->Conf->db);
	return(res);
}

#undef THINFO_TEST
#ifdef THINFO_TEST
/* CallBack function for Thread information */
void UdmThreadProc(int handle,char *state, char* str){
	printf("%d %s %s\n",handle,state,str);
}
#endif

static const char *csgroup(const UDM_CHARSET *cs) {
	switch(cs->family){
		case UDM_CHARSET_ARABIC		:	return "Arabic";
		case UDM_CHARSET_ARMENIAN	:	return "Armenian";
		case UDM_CHARSET_BALTIC		:	return "Baltic";
		case UDM_CHARSET_CELTIC		:	return "Celtic";
		case UDM_CHARSET_CENTRAL	:	return "Central Eur";
		case UDM_CHARSET_CHINESE_SIMPLIFIED:	return "Chinese Simplified";
		case UDM_CHARSET_CHINESE_TRADITIONAL:	return "Chinese Traditional";
		case UDM_CHARSET_CYRILLIC	:	return "Cyrillic";
		case UDM_CHARSET_GREEK		:	return "Greek";
		case UDM_CHARSET_HEBREW		:	return "Hebrew";
		case UDM_CHARSET_ICELANDIC	:	return "Icelandic";
		case UDM_CHARSET_JAPANESE	:	return "Japanese";
		case UDM_CHARSET_KOREAN		:	return "Korean";
		case UDM_CHARSET_NORDIC		:	return "Nordic";
		case UDM_CHARSET_SOUTHERN	:	return "South Eur";
		case UDM_CHARSET_THAI		:	return "Thai";
		case UDM_CHARSET_TURKISH	:	return "Turkish";
		case UDM_CHARSET_UNICODE	:	return "Unicode";
		case UDM_CHARSET_VIETNAMESE	:	return "Vietnamese";
		case UDM_CHARSET_WESTERN	:	return "Western";
		default				:	return "Unknown";
	}
}

static int cmpgrp(const void *v1, const void *v2){
	int res;
	const UDM_CHARSET *c1=v1;
	const UDM_CHARSET *c2=v2;
	if ((res=strcasecmp(csgroup(c1),csgroup(c2))))return res;
	return strcasecmp(c1->name,c2->name);
}

static void display_charsets(void){
	UDM_CHARSET *cs=NULL;
	UDM_CHARSET c[100];
	size_t i=0;
	size_t n=0;
	int family=-1;
	
	for(cs=UdmGetCharSetByID(0) ; cs && cs->name ; cs++){
		/* Skip not compiled charsets */
		if(cs->family != UDM_CHARSET_UNKNOWN)
			c[n++]=*cs;
	}
	fprintf(stderr,"\n%d charsets available:\n",n);

	qsort(c,n,sizeof(UDM_CHARSET),&cmpgrp);
	for(i=0;i<n;i++){
		if(family!=c[i].family){
			fprintf(stderr,"\n%11s : ",csgroup(&c[i]));
			family=c[i].family;
		}
		fprintf(stderr,"%s ",c[i].name);
	}
	fprintf(stderr,"\n");
}



static int usage(int level){

	fprintf(stderr,
"\n\
indexer from %s-%s-%s\n\
http://search.mnogo.ru (C) 1998-2002, LavTech Corp.\n\
\n\
Usage: indexer [OPTIONS]  [configfile]\n\
\n\
Indexing options:\
"
#ifdef HAVE_SQL
"\n\
  -a            reindex all documents even if not expired (may be\n\
                limited using -t, -u, -s, -c and -f options)\n\
  -m            reindex expired documents even if not modified (may\n\
                be limited using -t, -u, -c and -s options)\n\
  -e            index 'most expired' (oldest) documents first\n\
  -o            index documents with less depth (hops value) first\n\
  -n n          index only n documents and exit\n\
  -c n          index only n seconds and exit\n\
  -q            quick startup (do not add Server URLs)\n\
"
#endif
"\n\
  -b            block starting more indexing instance\n\
  -i            insert new URLs (URLs to insert must be given using -u or -f)\n\
  -p n          sleep n seconds after each URL\n\
  -w            do not warn before clearing documents from database\n\
"
#ifdef HAVE_PTHREAD
"  -N n          run N threads\n\
"
#endif

#ifdef HAVE_SQL
"\n\
Subsection control options (may be combined):\n\
  -s status     limit indexer to documents matching status (HTTP Status code)\n\
  -t tag        limit indexer to documents matching tag\n\
  -g category   limit indexer to documents matching category\n\
  -u pattern    limit indexer to documents with URLs matching pattern\n\
                (supports SQL LIKE wildcard '%%')\n\
  -f filename   read URLs to be indexed/inserted/cleared from file (with -a\n\
                or -C option, supports SQL LIKE wildcard '%%'; has no effect\n\
                when combined with -m option)\n\
  -f -          Use STDIN instead of file as URL list\n\
"
#else
"\n\
URL options:\n\
  -u URL        insert URL at startup\n\
  -f filename   read URLs to be inserted from file\n\
"
#endif
"\n\
Logging options:\n\
"
#ifdef LOG_PERROR
"  -l            do not log to stdout/stderr\n\
"
#endif
"  -v n          verbose level, 0-5\n\
\n\
Misc. options:\n\
"
#ifdef HAVE_SQL

"  -C            clear database and exit\n\
  -S            print statistics and exit\n\
  -I            print referers and exit\n\
"
#endif
"  -h,-?         print help page and exit\n\
  -hh           print more help and exit\n\
\n\
\n\
Please mail bug reports and suggestions to <general@mnogosearch.org>.\n",
	PACKAGE,VERSION,UDM_DBTYPE);
	
	if(level>1)display_charsets();
	return(0);
}



#if  (WIN32|WINNT)
DWORD WINAPI thread_main(void *arg){
#else
static void * thread_main(void *arg){
#endif
	UDM_AGENT * Indexer;
	int res=UDM_OK;
	int done=0;
	int i_sleep=0;
	
	UDM_MUTEX_LOCK(&mutex[UDM_LOCK_CONF]);
	Indexer=UdmAgentInit(NULL,&Conf, next_thread++);
	Indexer->flags=flags;
	UDM_MUTEX_UNLOCK(&mutex[UDM_LOCK_CONF]);

	while(!done){
	
		if(max_index_time>=0){
			time_t now;
			
			time(&now);
			if((now-Indexer->start_time)>max_index_time)
				break;
		}

		UDM_MUTEX_LOCK(&mutex[UDM_LOCK_CONF]);

		if(have_sighup){
			UDM_ENV	NewConf;
			int	rc;
			
			UdmEnvInit(&NewConf);
			UdmSetLockProc(&NewConf,UdmLockProc);
			UdmSetRefProc(&NewConf,UdmRefProc);

			UdmLog(Indexer,UDM_LOG_ERROR,"Reloading config '%s'",config_name);
			rc=UdmLoadConfig(&NewConf,config_name,0,add_servers+load_langmaps+UDM_FLAG_SPELL);
			
			if(rc!=UDM_OK){
				UdmLog(Indexer,UDM_LOG_ERROR,"Can't load config: %s",UdmEnvErrMsg(&NewConf));
				UdmLog(Indexer,UDM_LOG_ERROR,"Continuing with old config");
				UdmEnvFree(&NewConf);
			}else{
				UdmEnvFree(&Conf);
				Conf=NewConf;
				UdmOpenLog("indexer",&Conf, log2stderr);
			}

			have_sighup=0;
		}
		UDM_MUTEX_UNLOCK(&mutex[UDM_LOCK_CONF]);
		if(done)break;
		
		if(res==UDM_OK) /* Possible after bad startup */
			res=UdmIndexNextURL(Indexer);
		
		UDM_MUTEX_LOCK(&mutex[UDM_LOCK_CONF]);
		cur_url_number++;
		UDM_MUTEX_UNLOCK(&mutex[UDM_LOCK_CONF]);

		switch(res){
			case UDM_OK:
				if(i_sleep){
					UDM_MUTEX_LOCK(&mutex[UDM_LOCK_CONF]);
					sleep_threads--;
					UDM_MUTEX_UNLOCK(&mutex[UDM_LOCK_CONF]);
					i_sleep=0;
				}
				break;

			
			case UDM_NOTARGET:
#ifdef HAVE_PTHREAD
			/* in multi-threaded environment we		*/
			/* should wait for a moment when every thread	*/
			/* has nothing to do				*/

				if(!i_sleep){
					UDM_MUTEX_LOCK(&mutex[UDM_LOCK_CONF]);
					sleep_threads++;
					UDM_MUTEX_UNLOCK(&mutex[UDM_LOCK_CONF]);
					i_sleep=1;
				}

				UDM_MUTEX_LOCK(&mutex[UDM_LOCK_CONF]);
				done=(sleep_threads>=total_threads);
				UDM_MUTEX_UNLOCK(&mutex[UDM_LOCK_CONF]);

				break;
#else
				done=1;
				break;
#endif
			case UDM_ERROR:
			default:
#ifdef HAVE_PTHREAD
			/* in multi-threaded environment we		*/
			/* should wait for a moment when every thread	*/
			/* has nothing to do				*/

				if(!i_sleep){
				        UdmLog(Indexer,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Indexer->Conf));
					UDM_MUTEX_LOCK(&mutex[UDM_LOCK_CONF]);
					sleep_threads++;
					UDM_MUTEX_UNLOCK(&mutex[UDM_LOCK_CONF]);
					i_sleep=1;
				}

				UDM_MUTEX_LOCK(&mutex[UDM_LOCK_CONF]);
				done=(sleep_threads>=total_threads);
				UDM_MUTEX_UNLOCK(&mutex[UDM_LOCK_CONF]);

				break;
#else
				UdmLog(Indexer,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Indexer->Conf));
				done=1;
#endif
				break;
		}
		if((seconds)&&(!done)){
			UdmLog(Indexer,UDM_LOG_DEBUG,"Sleeping %d second(s)",seconds);
			UDMSLEEP(seconds);
		}
	}

	UDM_MUTEX_LOCK(&mutex[UDM_LOCK_CONF]);
	if(res!=UDM_ERROR){
		time_t now, sec;
		float M = 0.0, K = 0.0;
		time(&now);
		sec = now - Indexer->start_time;
		if (sec > 0) {
			M = Indexer->nbytes / 1048576.0 / sec;
			if (M < 1.0) K = Indexer->nbytes / 1024.0 / sec;
		}
		UdmLog(Indexer,UDM_LOG_ERROR,"Done (%d seconds, %d bytes, %5.2f %cbytes/sec.)",
			sec, Indexer->nbytes, (M < 1.0) ? K : M, (M < 1.0) ? 'K' : 'M' );
	}
	total_threads--;
	UDM_MUTEX_UNLOCK(&mutex[UDM_LOCK_CONF]);
	
	UdmAgentFree(Indexer);
	
#if     (WIN32|WINNT)
	return(0);
#else
	return(NULL);
#endif
}


static char pidname[1024];
static char time_pid[100];

static void exitproc(void){
	unlink(pidname);
}
static char * time_pid_info(void){
	struct tm * tim;
	time_t t;
	t=time(NULL);
	tim=localtime(&t);
	strftime(time_pid,sizeof(time_pid),"%a %d %T",tim);
	sprintf(time_pid+strlen(time_pid)," [%d]",(int)getpid());
	return(time_pid);
}

static void UdmWSAStartup(void){
#if (WIN32|WINNT)
	WSADATA wsaData;
	if(WSAStartup(0x101,&wsaData)!=0){
		fprintf(stderr,"WSAStartup() error %d\n",WSAGetLastError);
		exit(1);
	}
#endif
}

static void UdmWSACleanup(void){
#if (WIN32|WINNT)
	WSACleanup();
#endif
	return;
}


int main(int argc, char **argv) {
	char		*language=NULL,*affix=NULL,*dictionary=NULL;
	int		insert=0,clear=0,stat=0,integrity=0,expire=0;
	int		mkind=0, block=0;
	int		ch, pid_fd;
	int		warnings=1, maxthreads=1, help=0;
	char		*url_fname=NULL;
	char		pidbuf[1024];
	UDM_AGENT	Main;
	
	UdmWSAStartup();
	
	UdmInit(); /* Initialize library */
	
	InitMutexes();
	UdmEnvInit(&Conf);
	UdmSetLockProc(&Conf,UdmLockProc);
	UdmSetRefProc(&Conf,UdmRefProc);
#ifdef THINFO_TEST
	UdmSetThreadProc(&Conf,UdmThreadProc);
#endif
	
	while ((ch = getopt(argc, argv, "UCSIMabheolmdqiw?t:u:s:n:v:L:A:D:p:N:f:c:g:")) != -1){
		switch (ch) {
		case 'C': clear++;add_servers=0;load_langmaps=0;break;
		case 'S': stat++;add_servers=0;load_langmaps=0;break;
		case 'I': integrity++;add_servers=0;load_langmaps=0;break;
		case 'M': mkind=1;break;
		case 'q': add_servers=0;break;
		case 'l': log2stderr=0;break;
		case 'a': expire=1;break;
		case 'b': block++;break;
		case 'e': flags|=UDM_FLAG_SORT_EXPIRED;break;
		case 'o': flags|=UDM_FLAG_SORT_HOPS;break;
		case 'm': flags|=UDM_FLAG_REINDEX;break;
		case 'n': Conf.url_number=atoi(optarg);break;
		case 'c': max_index_time=atoi(optarg);break;
		case 'v': UdmSetLogLevel(&Conf, atoi(optarg));break;
		case 'p': seconds=atoi(optarg);break;
		case 't': UdmVarListAddStr(&Conf.Vars,"tag",optarg);break;
		case 'g': UdmVarListAddStr(&Conf.Vars,"cat",optarg);break;
		case 's': UdmVarListAddStr(&Conf.Vars,"s"  ,optarg);break;
		case 'u': UdmVarListAddStr(&Conf.Vars,"ul"  ,optarg);
			if(insert){
				UDM_HREF Href;
				Href.hops=0;
				Href.referrer=0;
				Href.url=optarg;
				Href.tag=NULL;
				Href.category=NULL;
				Href.stored=0;
				Href.method=UDM_METHOD_GET;
				UdmHrefListAdd(&Conf.Hrefs,&Href);
			}
			break;
		case 'N': maxthreads=atoi(optarg);break;
		case 'f': url_fname=optarg;break;
		case 'i': insert=1;break;
		case 'w': warnings=0;break;
		case '?':
		case 'h':
		default:
			help++;
		}
	}
	flags|=add_servers;

	argc -= optind;argv += optind;

	if((argc>1) || (help)){
		usage(help);
		UdmEnvFree(&Conf);
		return(1);
	}
	if(argc==1)config_name=argv[0];
	
	
	
	if(UDM_OK!=UdmLoadConfig(&Conf,config_name,0,add_servers+load_langmaps+((!integrity&&!stat&&!clear)*UDM_FLAG_SPELL))){
		fprintf(stderr,"%s\n",UdmEnvErrMsg(&Conf));
		UdmEnvFree(&Conf);
		exit(1);
	}
	
	
	
#if (WIN32|WINNT)
#else
	UdmOpenLog("indexer",&Conf, log2stderr);
	UdmSigHandlersInit(&Conf);
#endif
	
	UdmAgentInit(&Main,&Conf,0);
	Main.flags=flags;
	
	/* Open cachemode: FIXME: move to more proper place */
	if(UDM_OK!=UdmOpenCache(&Main,Conf.db)){
		fprintf(stderr,"Cache mode initializing error: '%s'\n",UdmEnvErrMsg(&Conf));
		UdmEnvFree(&Conf);
		exit(1);
	}
	
	if(url_fname && strcmp(url_fname,"-")) {
		/* Make sure URL file is readable if not STDIN */
		FILE *url_file;
		if(!(url_file=fopen(url_fname,"r"))){
			UdmLog(&Main,UDM_LOG_ERROR,"Error: can't open url file '%s': %s",url_fname, strerror(errno));
			goto ex;
		}
		fclose(url_file);
	}
	
	if(insert && url_fname) {
		
		if(strcmp(url_fname,"-")){
			/* Make sure all URLs to be inserted are OK */
			if(UDM_OK!=UdmURLFile(&Main, url_fname,UDM_URL_FILE_PARSE)){
				UdmLog(&Main,UDM_LOG_ERROR,"Error: Invalid URL in '%s'",url_fname);
				goto ex;
			}
		}
		
		if(UDM_OK!=UdmURLFile(&Main,url_fname,UDM_URL_FILE_INSERT)){
			UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
			goto ex;
		}
	}
	
	if(expire){
		int	res;
		
		if(url_fname){
			res=UdmURLFile(&Main,url_fname,UDM_URL_FILE_REINDEX);
		}else{
			res=UdmURLAction(&Main,NULL,UDM_URL_ACTION_EXPIRE,Main.Conf->db);
		}
		if(res!=UDM_OK){
			UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
			goto ex;
		}
	}
	
	if(affix||dictionary){
		if(!language){
			UdmLog(&Main,UDM_LOG_ERROR,"Error: Language is not specified for import!");
		}else
		if(strlen(language)!=2){
			UdmLog(&Main,UDM_LOG_ERROR,"Error: Language should be 2 letters!");
		}
		goto ex;
	}
	
	if(clear){
		int clear_confirmed=0;
		if(warnings) {
			char str[5]="";
			printf("You are going to delete database '%s' content\n",UdmVarListFindStr(&Conf.Vars,"DBAddr",""));
			printf("Are you sure?(YES/no)");
			if(fgets(str,sizeof(str),stdin))
				if(!strncmp(str,"YES",3))
					clear_confirmed=1;
		}
		else
			clear_confirmed=1;
		
		if(clear_confirmed) {
			if(url_fname) {
				if(UDM_OK!=UdmURLFile(&Main,url_fname,UDM_URL_FILE_CLEAR)){
					UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
				}
			}
			else {
				printf("Deleting...");
				if(UDM_OK!=UdmClearDatabase(&Main)){
					UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
				}
				printf("Done\n");
			}
		}else{
			printf("Canceled\n");
		}
	}else
	if(stat){
		if(UDM_OK!=ShowStatistics(&Main)){
			UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
		}
	}else
	if(integrity){
		if(UDM_OK!=ShowReferers(&Main)){
			UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
		}
	}else{
		if (block) {
			/* Check that another instance isn't running */
			/* and create PID file.                      */
			
			sprintf(pidname,"%s/%s",UDM_VAR_DIR,"indexer.pid");
			pid_fd = open(pidname,O_CREAT|O_EXCL|O_WRONLY,0644);
			if(pid_fd < 0){
				fprintf(stderr,"%s Can't create '%s': %s\n", time_pid_info(), pidname, strerror(errno));
				if(errno == EEXIST){
					fprintf(stderr,"It seems that another stored is already running!\n");
					fprintf(stderr,"Remove '%s' if it is not true.\n",pidname);
				}
				goto ex;
			}
			sprintf(pidbuf,"%d\n",(int)getpid());
			write(pid_fd,&pidbuf,strlen(pidbuf));
			atexit(&exitproc);
		}
		
		UdmLog(&Main,UDM_LOG_ERROR, "indexer from %s-%s-%s started with '%s'", PACKAGE, VERSION, UDM_DBTYPE, config_name);
		
#ifdef HAVE_PTHREAD
		{
#if (WIN32|WINNT)
#else
			pthread_t *threads=(pthread_t*)malloc(maxthreads*sizeof(pthread_t));
#endif
			
			int i;
			total_threads=maxthreads;
			
			for(i=0;i<maxthreads;i++){
#if (WIN32|WINNT)
				CreateThread(NULL, 0, &thread_main, NULL, 0, NULL);
#else
				pthread_attr_t attr;
				size_t stksize=1024*256;
				
				pthread_attr_init(&attr);
				pthread_attr_setstacksize(&attr, stksize);
				pthread_create(&threads[i],&attr,&thread_main,NULL);
#endif
			}
			
			while(1){
				int num;
				UDM_AGENT *A=&Main;
				UDM_GETLOCK(A,UDM_LOCK_CONF);
				num=total_threads;
				UDM_RELEASELOCK(A,UDM_LOCK_CONF);
				if(!num)break;
				UDMSLEEP(1);
			}
#if (WIN32|WINNT)
#else
			free(threads);
#endif
		}
#else
		thread_main(NULL);
#endif
	}
	
ex:
	total_threads=0;
	UdmAgentFree(&Main);
	UdmEnvFree(&Conf);
	DestroyMutexes();
	UdmWSACleanup();
	
	return(0);
}
